# Copyright 2022 VMware, Inc.
# All rights reserved. -- VMware Confidential

"""Functionalities to retrieve DPU info and attributes.
"""

from esxutils import EsxcliError, runCli

from ..Errors import DpuInfoError

import logging

log = logging.getLogger(__name__)

ALIAS = 'Device Alias'
ID = 'Id'
MANAGEMENT_STATE = 'Management State'
OS_VERSION = 'OS Version'
TRANSPORT_ADDR = 'Transport Address'

MANAGED = 'Managed'
UNAVAILABLE = 'Unavailable'
DEFAULT_OS_VERSION = 'N/A'

def getManagedDpuInfo():
   """Return a list of managed DPU info dicts.
      Example:
         [
            {
               'Address Family': 'IPv4',
               'Base PCI Address': '0000:0b:00.0',
               'Device Alias': 'vmdpu0'
               'Id': 'dpu0',
               'Local Endpoint Address': '10.184.108.241',
               'Management State': 'Managed',
               'Model': 'NVIDIA-mock',
               'OS Version': 'VMware ESXio 7.0.3'/'(N/A)'
               'Netstack Instance': 'defaultTcpipStack',
               'Primary': True,
               'Transport Address': '10.185.17.1',
               'Management State': 'Unavailable'/'Managed',
               'Vendor': 'VMware'
            }
         ]
   """
   try:
      dpuInfo = runCli(['hardwareinternal', 'dpu', 'list'], True)
      result = list()
      for dpu in dpuInfo:
         if dpu[MANAGEMENT_STATE] == MANAGED:
           result.append(dpu)
         elif dpu[MANAGEMENT_STATE] == UNAVAILABLE:
           if dpu[OS_VERSION] != DEFAULT_OS_VERSION:
              result.append(dpu)
           else:
              log.warning('Unavailable DPU %s without OS version is ignored',
                          dpu[ALIAS])
      return result
   except KeyError as e:
      raise DpuInfoError('Attribute %s not found' % str(e))
   except EsxcliError as e:
      raise DpuInfoError('Failed to get DPU list: %s' % str(e))

def getDpuID(dpuInfo):
   """ Helper function to get the DPU ID.
   """
   try:
      return dpuInfo[ID]
   except KeyError as e:
      msg = 'Failed to get ID from dpu info'
      raise DpuInfoError(msg, str(dpuInfo))

def getDpuTransportAddr(dpuInfo):
   """ Helper function to get DPU transport address.
   """
   try:
      return dpuInfo[TRANSPORT_ADDR]
   except KeyError as e:
      msg = 'Failed to get transport addr from dpu info'
      raise DpuInfoError(msg, str(dpuInfo))

def getDpuAlias(dpuInfo):
   """ Helper function to get Device Alias.
   """
   try:
      return dpuInfo[ALIAS]
   except KeyError as e:
      msg = 'Failed to get Device Alias from dpu info'
      raise DpuInfoError(msg, str(dpuInfo))

def getDpuInfoFromId(dpuList, dpuId):
   """ Helper function to retrieve Dpu Info from dpu id.
   """
   for dpu in dpuList:
      if dpuId == getDpuID(dpu):
         return dpu
   raise DpuInfoError('Failed to get DPU info for DPU id: %s' % str(dpuId))

def isDpuUnavailable(dpuInfo):
   """ Whether the dpu is in UNAVAILABLE state or not.
   """
   try:
      return dpuInfo[MANAGEMENT_STATE] == UNAVAILABLE
   except KeyError as e:
      return False
